#include "process.h"
#include "frame.h"
#include "symtab.h"
#include "symbol.h"
#include "core.h"
#include "sunossymtab.h"
#ifndef SUN4PI
#include "sunos5symtab.h"
#endif	/* SUN4PI */
#include "asm.h"
#include "sparcasm.h"
#include "sparccore.h"
SRCFILE("sparccore.c")

const long SPARC_UNIMPMASK = 0xC1C00000, SPARC_UNIMP = 0x00000000;
const long SPARC_CALLMASK = 0xC0000000, SPARC_CALL = 0x40000000;
const long SPARC_JMPLMSK = 0xFFF80000, SPARC_JMPL = 0x9FC00000;
const long SPARC_SAVESPM = 0xFFFFE000, SPARC_SAVESPIMM = 0x9DE3A000;
const long SPARC_SAVESPG1 = 0x9DE38001;

int SparcCore::REG_AP()			{ return 30; }
int SparcCore::REG_FP()			{ return 30; }
int SparcCore::REG_SP()			{ return 14; }
int SparcCore::REG_PC()			{ return 33; }
int SparcCore::REG_G1()			{ return 1; }
int SparcCore::REG_O0()			{ return 8; }
int SparcCore::REG_I7()			{ return 31; }
int SparcCore::REG_NPC()		{ return 34; }
int SparcCore::nregs()			{ return 36; }
int SparcCore::cntxtbytes()		{ return 19 << 2; }
long SparcCore::returnregloc()		{ return regloc(REG_O0()); }
Asm *SparcCore::newAsm()		{ return new SparcAsm(this); }

long SparcCore::callingpc(long)
{
	long l = regpeek(REG_I7()) + 8;
	if ((peek(l)->lng & SPARC_UNIMPMASK) == SPARC_UNIMP)
		l += 4;
	return l;
}

void SparcCore::newSymTab(long reloc)
{
#ifndef SUN4PI
	char magic[4];
	lseek(stabfd, 0L, 0);
	ReadOK(stabfd, (char*)magic, sizeof(magic));
	if (!strncmp(magic, "\177ELF", 4))
		 _symtab = new Sunos5SymTab(this, stabfd, _symtab, reloc);
	else
#endif	/* SUN4PI */
		 _symtab = new SunosSymTab(this, stabfd, _symtab, reloc);
}

SparcCore::SparcCore()
{
	stackdir = GROWDOWN;
	memlayout = MSBFIRST;
	bptsize = 4;
}

long SparcCore::saved(Frame *f, int r, int sz)	/* ignore size */
{
	if (r < 8)
		return 0;
	if (r < 16)
		return f->regbase + 4 * r + 4 - sz;
	if (r < 32 )
		return f->fp + 4 * (r - 16) + 4 - sz;
	return 0;
}

char *SparcCore::regname(int r)
{
	static char *regnames[] = {
		 "g0", "g1", "g2", "g3", "g4", "g5", "g6", "g7",
		 "o0", "o1", "o2", "o3", "o4", "o5", "sp", "o7",
		 "l0", "l1", "l2", "l3", "l4", "l5", "l6", "l7",
		 "i0", "i1", "i2", "i3", "i4", "i5", "fp", "i7",
		 "psr", "pc", "npc", "y",
	};
	if (r >= 0 && r < 36)
		return regnames[r];
	else
		return 0;
}

long SparcCore::regloc(int r, int sz)
{
	long ret;
	if (r < 1) {
		if (r == 0)
			ret = regaddr() + 4 * 19;
		else
			return 0;
	} else if (r < 16)	// %g1 - %o7 */
		ret = regaddr() + 4 * r + 12;
	else if (r < 32)	// %l0 - %i7
		ret = sp() + 4 * (r - 16);
	else if (r < 36)	// %psr - %y
		ret = regaddr() + 4 * (r - 32);
	else
		return 0;
	if (sz && sz < 4)
		ret += 4 - sz;
	return ret;
}

/*
 * This is slightly messy because the saved pc is referenced relative
 * to the sp of the stack frame below, instead of the fp.
 * Since sp(i) is equivalent to fp(i+1) we actually store 
 * the fp of the previous stack frame
 * in callstack.fpf[i].fp, instead of fp, so SparcCore::frameabove can
 * calculate the calling pc correctly.
 */
CallStk *SparcCore::callstack()	// do we have to peek everything twice?
{
	long _fp = fp();
	if( !fpvalid(_fp))
		return (CallStk *)0;
	for(long size = 1; size<1000; ++size ){
		long __fp = peek(_fp+56)->lng;
		if( !instack(__fp, _fp) )
			break;
		_fp = __fp;
	}
	CallStk *c = new CallStk(size, this);
	long _pc = pc();
	_fp = sp();
	for( long i = 0;;){
		c->fpf[i].fp = _fp;
		c->fpf[i].func = (Func*) _symtab->loctosym(U_FUNC, _pc);
		if (++i == size)
			break;
		_pc = peek(_fp+60)->lng;
		_fp = peek(_fp+56)->lng;
	}
	return c;
}

Frame SparcCore::frameabove(long _fp)
{
	Frame f(this);
	if ( _fp) {
		f.pc = peek(_fp+60)->lng;
		f.regbase = peek(_fp+56)->lng;
		f.fp = peek(f.regbase + 56)->lng;
	} else {
		f.pc = pc();
		f.fp = fp();
		f.regbase = sp();
	}
	f.ap = f.fp;
	return f;
}

long SparcCore::instrafterjsr()
{
	long retaddr = pc()+8;
	long inst = peek(retaddr)->lng;
	if ((inst & SPARC_UNIMPMASK) == SPARC_UNIMP)
		retaddr += 4;
	return retaddr;
}

int SparcCore::atjsr(long pc)
{
	long inst = peek(pc)->lng;
	return ( (inst & SPARC_CALLMASK) == SPARC_CALL ||
		 (inst & SPARC_JMPLMSK) == SPARC_JMPL );
}

char *SparcCore::stepprolog()
{
	Stmt *s;
	long lastpc, cpc;
	lastpc = cpc = pc();
	Func *f = (Func*)_symtab->loctosym(U_FUNC, cpc);
	if (f && (s = f->blk()->stmt)) {
		if (cpc < s->range.lo)
			lastpc = s->range.lo;
		else if (cpc == s->range.lo)
			lastpc = s->range.hi;
	}
	if (lastpc == cpc) {
		if ((peek(cpc)->lng & SPARC_SAVESPM) == SPARC_SAVESPIMM)
			lastpc += 4;
		else if (peek(cpc+8)->lng == SPARC_SAVESPG1)
			lastpc += 12;
	}
	return lastpc == cpc ? 0 : step(cpc, lastpc);
}

char *SparcCore::docall(long addr, int /*numarg*/)
{
	long save[3];
	char *error;
	static long newinstr[] = {
		0x9FC04000,	// call %g1+%g0,%o7
		0x01000000,	// nop
		0x01000000,	// nop
	};

	if( !online() )
		return "cannot restart dump";
	if( behavetype() == ACTIVE )
		return "process not stopped";
	long callstart = scratchaddr();
	if( ( error = read(callstart, (char*)save, sizeof(save)) )
	 || ( error = regpoke(REG_G1(), addr) )
	 || ( error = regpoke(REG_PC(), callstart) )
	 || ( error = regpoke(REG_NPC(), callstart+4) )
	 || ( error = write(callstart, (char*)newinstr, sizeof(newinstr)) )
	 || ( error = step(callstart, callstart+sizeof(save)) )
	 || ( error = write(callstart, (char*)save, sizeof(save)) )  )
		return error;
	return 0;
}

long SparcCore::apforcall(int argbytes)
{
	if (argbytes > 24)	// Sorry, only 6(*4) OUT regs
		return 0;
	return (regloc(REG_O0()) - 0x44); // so arg0 ends up in O0
}
